<?php

namespace Square\Types;

use Square\Core\Json\JsonSerializableType;
use Square\Core\Json\JsonProperty;
use Square\Core\Types\ArrayType;
use Square\Core\Types\Union;

/**
 * Links an order line item to a fulfillment. Each entry must reference
 * a valid `uid` for an order line item in the `line_item_uid` field, as well as a `quantity` to
 * fulfill.
 */
class FulfillmentFulfillmentEntry extends JsonSerializableType
{
    /**
     * @var ?string $uid A unique ID that identifies the fulfillment entry only within this order.
     */
    #[JsonProperty('uid')]
    private ?string $uid;

    /**
     * @var string $lineItemUid The `uid` from the order line item.
     */
    #[JsonProperty('line_item_uid')]
    private string $lineItemUid;

    /**
     * The quantity of the line item being fulfilled, formatted as a decimal number.
     * For example, `"3"`.
     *
     * Fulfillments for line items with a `quantity_unit` can have non-integer quantities.
     * For example, `"1.70000"`.
     *
     * @var string $quantity
     */
    #[JsonProperty('quantity')]
    private string $quantity;

    /**
     * Application-defined data attached to this fulfillment entry. Metadata fields are intended
     * to store descriptive references or associations with an entity in another system or store brief
     * information about the object. Square does not process this field; it only stores and returns it
     * in relevant API calls. Do not use metadata to store any sensitive information (such as personally
     * identifiable information or card details).
     *
     * Keys written by applications must be 60 characters or less and must be in the character set
     * `[a-zA-Z0-9_-]`. Entries can also include metadata generated by Square. These keys are prefixed
     * with a namespace, separated from the key with a ':' character.
     *
     * Values have a maximum length of 255 characters.
     *
     * An application can have up to 10 entries per metadata field.
     *
     * Entries written by applications are private and can only be read or modified by the same
     * application.
     *
     * For more information, see [Metadata](https://developer.squareup.com/docs/build-basics/metadata).
     *
     * @var ?array<string, ?string> $metadata
     */
    #[JsonProperty('metadata'), ArrayType(['string' => new Union('string', 'null')])]
    private ?array $metadata;

    /**
     * @param array{
     *   lineItemUid: string,
     *   quantity: string,
     *   uid?: ?string,
     *   metadata?: ?array<string, ?string>,
     * } $values
     */
    public function __construct(
        array $values,
    ) {
        $this->uid = $values['uid'] ?? null;
        $this->lineItemUid = $values['lineItemUid'];
        $this->quantity = $values['quantity'];
        $this->metadata = $values['metadata'] ?? null;
    }

    /**
     * @return ?string
     */
    public function getUid(): ?string
    {
        return $this->uid;
    }

    /**
     * @param ?string $value
     */
    public function setUid(?string $value = null): self
    {
        $this->uid = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function getLineItemUid(): string
    {
        return $this->lineItemUid;
    }

    /**
     * @param string $value
     */
    public function setLineItemUid(string $value): self
    {
        $this->lineItemUid = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function getQuantity(): string
    {
        return $this->quantity;
    }

    /**
     * @param string $value
     */
    public function setQuantity(string $value): self
    {
        $this->quantity = $value;
        return $this;
    }

    /**
     * @return ?array<string, ?string>
     */
    public function getMetadata(): ?array
    {
        return $this->metadata;
    }

    /**
     * @param ?array<string, ?string> $value
     */
    public function setMetadata(?array $value = null): self
    {
        $this->metadata = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        return $this->toJson();
    }
}
