<?php

namespace Square\Types;

use Square\Core\Json\JsonSerializableType;
use Square\Core\Json\JsonProperty;
use Square\Core\Types\ArrayType;
use Square\Core\Types\Union;

/**
 * Represents a discount that applies to one or more line items in an
 * order.
 *
 * Fixed-amount, order-scoped discounts are distributed across all non-zero line item totals.
 * The amount distributed to each line item is relative to the
 * amount contributed by the item to the order subtotal.
 */
class OrderLineItemDiscount extends JsonSerializableType
{
    /**
     * @var ?string $uid A unique ID that identifies the discount only within this order.
     */
    #[JsonProperty('uid')]
    private ?string $uid;

    /**
     * @var ?string $catalogObjectId The catalog object ID referencing [CatalogDiscount](entity:CatalogDiscount).
     */
    #[JsonProperty('catalog_object_id')]
    private ?string $catalogObjectId;

    /**
     * @var ?int $catalogVersion The version of the catalog object that this discount references.
     */
    #[JsonProperty('catalog_version')]
    private ?int $catalogVersion;

    /**
     * @var ?string $name The discount's name.
     */
    #[JsonProperty('name')]
    private ?string $name;

    /**
     * The type of the discount.
     *
     * Discounts that do not reference a catalog object ID must have a type of
     * `FIXED_PERCENTAGE` or `FIXED_AMOUNT`.
     * See [OrderLineItemDiscountType](#type-orderlineitemdiscounttype) for possible values
     *
     * @var ?value-of<OrderLineItemDiscountType> $type
     */
    #[JsonProperty('type')]
    private ?string $type;

    /**
     * The percentage of the discount, as a string representation of a decimal number.
     * A value of `7.25` corresponds to a percentage of 7.25%.
     *
     * `percentage` is not set for amount-based discounts.
     *
     * @var ?string $percentage
     */
    #[JsonProperty('percentage')]
    private ?string $percentage;

    /**
     * The total declared monetary amount of the discount.
     *
     * `amount_money` is not set for percentage-based discounts.
     *
     * @var ?Money $amountMoney
     */
    #[JsonProperty('amount_money')]
    private ?Money $amountMoney;

    /**
     * The amount of discount actually applied to the line item.
     *
     * The amount represents the amount of money applied as a line-item scoped discount.
     * When an amount-based discount is scoped to the entire order, the value
     * of `applied_money` is different than `amount_money` because the total
     * amount of the discount is distributed across all line items.
     *
     * @var ?Money $appliedMoney
     */
    #[JsonProperty('applied_money')]
    private ?Money $appliedMoney;

    /**
     * Application-defined data attached to this discount. Metadata fields are intended
     * to store descriptive references or associations with an entity in another system or store brief
     * information about the object. Square does not process this field; it only stores and returns it
     * in relevant API calls. Do not use metadata to store any sensitive information (such as personally
     * identifiable information or card details).
     *
     * Keys written by applications must be 60 characters or less and must be in the character set
     * `[a-zA-Z0-9_-]`. Entries can also include metadata generated by Square. These keys are prefixed
     * with a namespace, separated from the key with a ':' character.
     *
     * Values have a maximum length of 255 characters.
     *
     * An application can have up to 10 entries per metadata field.
     *
     * Entries written by applications are private and can only be read or modified by the same
     * application.
     *
     * For more information, see [Metadata](https://developer.squareup.com/docs/build-basics/metadata).
     *
     * @var ?array<string, ?string> $metadata
     */
    #[JsonProperty('metadata'), ArrayType(['string' => new Union('string', 'null')])]
    private ?array $metadata;

    /**
     * Indicates the level at which the discount applies. For `ORDER` scoped discounts,
     * Square generates references in `applied_discounts` on all order line items that do
     * not have them. For `LINE_ITEM` scoped discounts, the discount only applies to line items
     * with a discount reference in their `applied_discounts` field.
     *
     * This field is immutable. To change the scope of a discount, you must delete
     * the discount and re-add it as a new discount.
     * See [OrderLineItemDiscountScope](#type-orderlineitemdiscountscope) for possible values
     *
     * @var ?value-of<OrderLineItemDiscountScope> $scope
     */
    #[JsonProperty('scope')]
    private ?string $scope;

    /**
     * The reward IDs corresponding to this discount. The application and
     * specification of discounts that have `reward_ids` are completely controlled by the backing
     * criteria corresponding to the reward tiers of the rewards that are added to the order
     * through the Loyalty API. To manually unapply discounts that are the result of added rewards,
     * the rewards must be removed from the order through the Loyalty API.
     *
     * @var ?array<string> $rewardIds
     */
    #[JsonProperty('reward_ids'), ArrayType(['string'])]
    private ?array $rewardIds;

    /**
     * The object ID of a [pricing rule](entity:CatalogPricingRule) to be applied
     * automatically to this discount. The specification and application of the discounts, to
     * which a `pricing_rule_id` is assigned, are completely controlled by the corresponding
     * pricing rule.
     *
     * @var ?string $pricingRuleId
     */
    #[JsonProperty('pricing_rule_id')]
    private ?string $pricingRuleId;

    /**
     * @param array{
     *   uid?: ?string,
     *   catalogObjectId?: ?string,
     *   catalogVersion?: ?int,
     *   name?: ?string,
     *   type?: ?value-of<OrderLineItemDiscountType>,
     *   percentage?: ?string,
     *   amountMoney?: ?Money,
     *   appliedMoney?: ?Money,
     *   metadata?: ?array<string, ?string>,
     *   scope?: ?value-of<OrderLineItemDiscountScope>,
     *   rewardIds?: ?array<string>,
     *   pricingRuleId?: ?string,
     * } $values
     */
    public function __construct(
        array $values = [],
    ) {
        $this->uid = $values['uid'] ?? null;
        $this->catalogObjectId = $values['catalogObjectId'] ?? null;
        $this->catalogVersion = $values['catalogVersion'] ?? null;
        $this->name = $values['name'] ?? null;
        $this->type = $values['type'] ?? null;
        $this->percentage = $values['percentage'] ?? null;
        $this->amountMoney = $values['amountMoney'] ?? null;
        $this->appliedMoney = $values['appliedMoney'] ?? null;
        $this->metadata = $values['metadata'] ?? null;
        $this->scope = $values['scope'] ?? null;
        $this->rewardIds = $values['rewardIds'] ?? null;
        $this->pricingRuleId = $values['pricingRuleId'] ?? null;
    }

    /**
     * @return ?string
     */
    public function getUid(): ?string
    {
        return $this->uid;
    }

    /**
     * @param ?string $value
     */
    public function setUid(?string $value = null): self
    {
        $this->uid = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getCatalogObjectId(): ?string
    {
        return $this->catalogObjectId;
    }

    /**
     * @param ?string $value
     */
    public function setCatalogObjectId(?string $value = null): self
    {
        $this->catalogObjectId = $value;
        return $this;
    }

    /**
     * @return ?int
     */
    public function getCatalogVersion(): ?int
    {
        return $this->catalogVersion;
    }

    /**
     * @param ?int $value
     */
    public function setCatalogVersion(?int $value = null): self
    {
        $this->catalogVersion = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @param ?string $value
     */
    public function setName(?string $value = null): self
    {
        $this->name = $value;
        return $this;
    }

    /**
     * @return ?value-of<OrderLineItemDiscountType>
     */
    public function getType(): ?string
    {
        return $this->type;
    }

    /**
     * @param ?value-of<OrderLineItemDiscountType> $value
     */
    public function setType(?string $value = null): self
    {
        $this->type = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getPercentage(): ?string
    {
        return $this->percentage;
    }

    /**
     * @param ?string $value
     */
    public function setPercentage(?string $value = null): self
    {
        $this->percentage = $value;
        return $this;
    }

    /**
     * @return ?Money
     */
    public function getAmountMoney(): ?Money
    {
        return $this->amountMoney;
    }

    /**
     * @param ?Money $value
     */
    public function setAmountMoney(?Money $value = null): self
    {
        $this->amountMoney = $value;
        return $this;
    }

    /**
     * @return ?Money
     */
    public function getAppliedMoney(): ?Money
    {
        return $this->appliedMoney;
    }

    /**
     * @param ?Money $value
     */
    public function setAppliedMoney(?Money $value = null): self
    {
        $this->appliedMoney = $value;
        return $this;
    }

    /**
     * @return ?array<string, ?string>
     */
    public function getMetadata(): ?array
    {
        return $this->metadata;
    }

    /**
     * @param ?array<string, ?string> $value
     */
    public function setMetadata(?array $value = null): self
    {
        $this->metadata = $value;
        return $this;
    }

    /**
     * @return ?value-of<OrderLineItemDiscountScope>
     */
    public function getScope(): ?string
    {
        return $this->scope;
    }

    /**
     * @param ?value-of<OrderLineItemDiscountScope> $value
     */
    public function setScope(?string $value = null): self
    {
        $this->scope = $value;
        return $this;
    }

    /**
     * @return ?array<string>
     */
    public function getRewardIds(): ?array
    {
        return $this->rewardIds;
    }

    /**
     * @param ?array<string> $value
     */
    public function setRewardIds(?array $value = null): self
    {
        $this->rewardIds = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getPricingRuleId(): ?string
    {
        return $this->pricingRuleId;
    }

    /**
     * @param ?string $value
     */
    public function setPricingRuleId(?string $value = null): self
    {
        $this->pricingRuleId = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        return $this->toJson();
    }
}
