<?php

namespace Square\Payments\Requests;

use Square\Core\Json\JsonSerializableType;
use Square\Core\Json\JsonProperty;
use Square\Types\Money;
use Square\Types\Address;
use Square\Types\CashPaymentDetails;
use Square\Types\ExternalPaymentDetails;
use Square\Types\CustomerDetails;
use Square\Types\OfflinePaymentDetails;

class CreatePaymentRequest extends JsonSerializableType
{
    /**
     * The ID for the source of funds for this payment.
     * This could be a payment token generated by the Web Payments SDK for any of its
     * [supported methods](https://developer.squareup.com/docs/web-payments/overview#explore-payment-methods),
     * including cards, bank transfers, Afterpay or Cash App Pay. If recording a payment
     * that the seller received outside of Square, specify either "CASH" or "EXTERNAL".
     * For more information, see
     * [Take Payments](https://developer.squareup.com/docs/payments-api/take-payments).
     *
     * @var string $sourceId
     */
    #[JsonProperty('source_id')]
    private string $sourceId;

    /**
     * A unique string that identifies this `CreatePayment` request. Keys can be any valid string
     * but must be unique for every `CreatePayment` request.
     *
     * Note: The number of allowed characters might be less than the stated maximum, if multi-byte
     * characters are used.
     *
     * For more information, see [Idempotency](https://developer.squareup.com/docs/working-with-apis/idempotency).
     *
     * @var string $idempotencyKey
     */
    #[JsonProperty('idempotency_key')]
    private string $idempotencyKey;

    /**
     * The amount of money to accept for this payment, not including `tip_money`.
     *
     * The amount must be specified in the smallest denomination of the applicable currency
     * (for example, US dollar amounts are specified in cents). For more information, see
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-monetary-amounts).
     *
     * The currency code must match the currency associated with the business
     * that is accepting the payment.
     *
     * @var ?Money $amountMoney
     */
    #[JsonProperty('amount_money')]
    private ?Money $amountMoney;

    /**
     * The amount designated as a tip, in addition to `amount_money`.
     *
     * The amount must be specified in the smallest denomination of the applicable currency
     * (for example, US dollar amounts are specified in cents). For more information, see
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-monetary-amounts).
     *
     * The currency code must match the currency associated with the business
     * that is accepting the payment.
     *
     * @var ?Money $tipMoney
     */
    #[JsonProperty('tip_money')]
    private ?Money $tipMoney;

    /**
     * The amount of money that the developer is taking as a fee
     * for facilitating the payment on behalf of the seller.
     *
     * The amount cannot be more than 90% of the total amount of the payment.
     *
     * The amount must be specified in the smallest denomination of the applicable currency
     * (for example, US dollar amounts are specified in cents). For more information, see
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-monetary-amounts).
     *
     * The fee currency code must match the currency associated with the seller
     * that is accepting the payment. The application must be from a developer
     * account in the same country and using the same currency code as the seller.
     *
     * For more information about the application fee scenario, see
     * [Take Payments and Collect Fees](https://developer.squareup.com/docs/payments-api/take-payments-and-collect-fees).
     *
     * To set this field, `PAYMENTS_WRITE_ADDITIONAL_RECIPIENTS` OAuth permission is required.
     * For more information, see [Permissions](https://developer.squareup.com/docs/payments-api/take-payments-and-collect-fees#permissions).
     *
     * @var ?Money $appFeeMoney
     */
    #[JsonProperty('app_fee_money')]
    private ?Money $appFeeMoney;

    /**
     * The duration of time after the payment's creation when Square automatically
     * either completes or cancels the payment depending on the `delay_action` field value.
     * For more information, see
     * [Time threshold](https://developer.squareup.com/docs/payments-api/take-payments/card-payments/delayed-capture#time-threshold).
     *
     * This parameter should be specified as a time duration, in RFC 3339 format.
     *
     * Note: This feature is only supported for card payments. This parameter can only be set for a delayed
     * capture payment (`autocomplete=false`).
     *
     * Default:
     *
     * - Card-present payments: "PT36H" (36 hours) from the creation time.
     * - Card-not-present payments: "P7D" (7 days) from the creation time.
     *
     * @var ?string $delayDuration
     */
    #[JsonProperty('delay_duration')]
    private ?string $delayDuration;

    /**
     * The action to be applied to the payment when the `delay_duration` has elapsed. The action must be
     * CANCEL or COMPLETE. For more information, see
     * [Time Threshold](https://developer.squareup.com/docs/payments-api/take-payments/card-payments/delayed-capture#time-threshold).
     *
     * Default: CANCEL
     *
     * @var ?string $delayAction
     */
    #[JsonProperty('delay_action')]
    private ?string $delayAction;

    /**
     * If set to `true`, this payment will be completed when possible. If
     * set to `false`, this payment is held in an approved state until either
     * explicitly completed (captured) or canceled (voided). For more information, see
     * [Delayed capture](https://developer.squareup.com/docs/payments-api/take-payments/card-payments#delayed-capture-of-a-card-payment).
     *
     * Default: true
     *
     * @var ?bool $autocomplete
     */
    #[JsonProperty('autocomplete')]
    private ?bool $autocomplete;

    /**
     * @var ?string $orderId Associates a previously created order with this payment.
     */
    #[JsonProperty('order_id')]
    private ?string $orderId;

    /**
     * The [Customer](entity:Customer) ID of the customer associated with the payment.
     *
     * This is required if the `source_id` refers to a card on file created using the Cards API.
     *
     * @var ?string $customerId
     */
    #[JsonProperty('customer_id')]
    private ?string $customerId;

    /**
     * The location ID to associate with the payment. If not specified, the [main location](https://developer.squareup.com/docs/locations-api#about-the-main-location) is
     * used.
     *
     * @var ?string $locationId
     */
    #[JsonProperty('location_id')]
    private ?string $locationId;

    /**
     * An optional [TeamMember](entity:TeamMember) ID to associate with
     * this payment.
     *
     * @var ?string $teamMemberId
     */
    #[JsonProperty('team_member_id')]
    private ?string $teamMemberId;

    /**
     * A user-defined ID to associate with the payment.
     *
     * You can use this field to associate the payment to an entity in an external system
     * (for example, you might specify an order ID that is generated by a third-party shopping cart).
     *
     * @var ?string $referenceId
     */
    #[JsonProperty('reference_id')]
    private ?string $referenceId;

    /**
     * An identifying token generated by [payments.verifyBuyer()](https://developer.squareup.com/reference/sdks/web/payments/objects/Payments#Payments.verifyBuyer).
     * Verification tokens encapsulate customer device information and 3-D Secure
     * challenge results to indicate that Square has verified the buyer identity.
     *
     * For more information, see [SCA Overview](https://developer.squareup.com/docs/sca-overview).
     *
     * @var ?string $verificationToken
     */
    #[JsonProperty('verification_token')]
    private ?string $verificationToken;

    /**
     * If set to `true` and charging a Square Gift Card, a payment might be returned with
     * `amount_money` equal to less than what was requested. For example, a request for $20 when charging
     * a Square Gift Card with a balance of $5 results in an APPROVED payment of $5. You might choose
     * to prompt the buyer for an additional payment to cover the remainder or cancel the Gift Card
     * payment. This field cannot be `true` when `autocomplete = true`.
     *
     * For more information, see
     * [Partial amount with Square Gift Cards](https://developer.squareup.com/docs/payments-api/take-payments#partial-payment-gift-card).
     *
     * Default: false
     *
     * @var ?bool $acceptPartialAuthorization
     */
    #[JsonProperty('accept_partial_authorization')]
    private ?bool $acceptPartialAuthorization;

    /**
     * @var ?string $buyerEmailAddress The buyer's email address.
     */
    #[JsonProperty('buyer_email_address')]
    private ?string $buyerEmailAddress;

    /**
     * The buyer's phone number.
     * Must follow the following format:
     * 1. A leading + symbol (followed by a country code)
     * 2. The phone number can contain spaces and the special characters `(` , `)` , `-` , and `.`.
     * Alphabetical characters aren't allowed.
     * 3. The phone number must contain between 9 and 16 digits.
     *
     * @var ?string $buyerPhoneNumber
     */
    #[JsonProperty('buyer_phone_number')]
    private ?string $buyerPhoneNumber;

    /**
     * @var ?Address $billingAddress The buyer's billing address.
     */
    #[JsonProperty('billing_address')]
    private ?Address $billingAddress;

    /**
     * @var ?Address $shippingAddress The buyer's shipping address.
     */
    #[JsonProperty('shipping_address')]
    private ?Address $shippingAddress;

    /**
     * @var ?string $note An optional note to be entered by the developer when creating a payment.
     */
    #[JsonProperty('note')]
    private ?string $note;

    /**
     * Optional additional payment information to include on the customer's card statement
     * as part of the statement description. This can be, for example, an invoice number, ticket number,
     * or short description that uniquely identifies the purchase.
     *
     * Note that the `statement_description_identifier` might get truncated on the statement description
     * to fit the required information including the Square identifier (SQ *) and name of the
     * seller taking the payment.
     *
     * @var ?string $statementDescriptionIdentifier
     */
    #[JsonProperty('statement_description_identifier')]
    private ?string $statementDescriptionIdentifier;

    /**
     * @var ?CashPaymentDetails $cashDetails Additional details required when recording a cash payment (`source_id` is CASH).
     */
    #[JsonProperty('cash_details')]
    private ?CashPaymentDetails $cashDetails;

    /**
     * @var ?ExternalPaymentDetails $externalDetails Additional details required when recording an external payment (`source_id` is EXTERNAL).
     */
    #[JsonProperty('external_details')]
    private ?ExternalPaymentDetails $externalDetails;

    /**
     * @var ?CustomerDetails $customerDetails Details about the customer making the payment.
     */
    #[JsonProperty('customer_details')]
    private ?CustomerDetails $customerDetails;

    /**
     * An optional field for specifying the offline payment details. This is intended for
     * internal 1st-party callers only.
     *
     * @var ?OfflinePaymentDetails $offlinePaymentDetails
     */
    #[JsonProperty('offline_payment_details')]
    private ?OfflinePaymentDetails $offlinePaymentDetails;

    /**
     * @param array{
     *   sourceId: string,
     *   idempotencyKey: string,
     *   amountMoney?: ?Money,
     *   tipMoney?: ?Money,
     *   appFeeMoney?: ?Money,
     *   delayDuration?: ?string,
     *   delayAction?: ?string,
     *   autocomplete?: ?bool,
     *   orderId?: ?string,
     *   customerId?: ?string,
     *   locationId?: ?string,
     *   teamMemberId?: ?string,
     *   referenceId?: ?string,
     *   verificationToken?: ?string,
     *   acceptPartialAuthorization?: ?bool,
     *   buyerEmailAddress?: ?string,
     *   buyerPhoneNumber?: ?string,
     *   billingAddress?: ?Address,
     *   shippingAddress?: ?Address,
     *   note?: ?string,
     *   statementDescriptionIdentifier?: ?string,
     *   cashDetails?: ?CashPaymentDetails,
     *   externalDetails?: ?ExternalPaymentDetails,
     *   customerDetails?: ?CustomerDetails,
     *   offlinePaymentDetails?: ?OfflinePaymentDetails,
     * } $values
     */
    public function __construct(
        array $values,
    ) {
        $this->sourceId = $values['sourceId'];
        $this->idempotencyKey = $values['idempotencyKey'];
        $this->amountMoney = $values['amountMoney'] ?? null;
        $this->tipMoney = $values['tipMoney'] ?? null;
        $this->appFeeMoney = $values['appFeeMoney'] ?? null;
        $this->delayDuration = $values['delayDuration'] ?? null;
        $this->delayAction = $values['delayAction'] ?? null;
        $this->autocomplete = $values['autocomplete'] ?? null;
        $this->orderId = $values['orderId'] ?? null;
        $this->customerId = $values['customerId'] ?? null;
        $this->locationId = $values['locationId'] ?? null;
        $this->teamMemberId = $values['teamMemberId'] ?? null;
        $this->referenceId = $values['referenceId'] ?? null;
        $this->verificationToken = $values['verificationToken'] ?? null;
        $this->acceptPartialAuthorization = $values['acceptPartialAuthorization'] ?? null;
        $this->buyerEmailAddress = $values['buyerEmailAddress'] ?? null;
        $this->buyerPhoneNumber = $values['buyerPhoneNumber'] ?? null;
        $this->billingAddress = $values['billingAddress'] ?? null;
        $this->shippingAddress = $values['shippingAddress'] ?? null;
        $this->note = $values['note'] ?? null;
        $this->statementDescriptionIdentifier = $values['statementDescriptionIdentifier'] ?? null;
        $this->cashDetails = $values['cashDetails'] ?? null;
        $this->externalDetails = $values['externalDetails'] ?? null;
        $this->customerDetails = $values['customerDetails'] ?? null;
        $this->offlinePaymentDetails = $values['offlinePaymentDetails'] ?? null;
    }

    /**
     * @return string
     */
    public function getSourceId(): string
    {
        return $this->sourceId;
    }

    /**
     * @param string $value
     */
    public function setSourceId(string $value): self
    {
        $this->sourceId = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function getIdempotencyKey(): string
    {
        return $this->idempotencyKey;
    }

    /**
     * @param string $value
     */
    public function setIdempotencyKey(string $value): self
    {
        $this->idempotencyKey = $value;
        return $this;
    }

    /**
     * @return ?Money
     */
    public function getAmountMoney(): ?Money
    {
        return $this->amountMoney;
    }

    /**
     * @param ?Money $value
     */
    public function setAmountMoney(?Money $value = null): self
    {
        $this->amountMoney = $value;
        return $this;
    }

    /**
     * @return ?Money
     */
    public function getTipMoney(): ?Money
    {
        return $this->tipMoney;
    }

    /**
     * @param ?Money $value
     */
    public function setTipMoney(?Money $value = null): self
    {
        $this->tipMoney = $value;
        return $this;
    }

    /**
     * @return ?Money
     */
    public function getAppFeeMoney(): ?Money
    {
        return $this->appFeeMoney;
    }

    /**
     * @param ?Money $value
     */
    public function setAppFeeMoney(?Money $value = null): self
    {
        $this->appFeeMoney = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getDelayDuration(): ?string
    {
        return $this->delayDuration;
    }

    /**
     * @param ?string $value
     */
    public function setDelayDuration(?string $value = null): self
    {
        $this->delayDuration = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getDelayAction(): ?string
    {
        return $this->delayAction;
    }

    /**
     * @param ?string $value
     */
    public function setDelayAction(?string $value = null): self
    {
        $this->delayAction = $value;
        return $this;
    }

    /**
     * @return ?bool
     */
    public function getAutocomplete(): ?bool
    {
        return $this->autocomplete;
    }

    /**
     * @param ?bool $value
     */
    public function setAutocomplete(?bool $value = null): self
    {
        $this->autocomplete = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getOrderId(): ?string
    {
        return $this->orderId;
    }

    /**
     * @param ?string $value
     */
    public function setOrderId(?string $value = null): self
    {
        $this->orderId = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getCustomerId(): ?string
    {
        return $this->customerId;
    }

    /**
     * @param ?string $value
     */
    public function setCustomerId(?string $value = null): self
    {
        $this->customerId = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getLocationId(): ?string
    {
        return $this->locationId;
    }

    /**
     * @param ?string $value
     */
    public function setLocationId(?string $value = null): self
    {
        $this->locationId = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getTeamMemberId(): ?string
    {
        return $this->teamMemberId;
    }

    /**
     * @param ?string $value
     */
    public function setTeamMemberId(?string $value = null): self
    {
        $this->teamMemberId = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getReferenceId(): ?string
    {
        return $this->referenceId;
    }

    /**
     * @param ?string $value
     */
    public function setReferenceId(?string $value = null): self
    {
        $this->referenceId = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getVerificationToken(): ?string
    {
        return $this->verificationToken;
    }

    /**
     * @param ?string $value
     */
    public function setVerificationToken(?string $value = null): self
    {
        $this->verificationToken = $value;
        return $this;
    }

    /**
     * @return ?bool
     */
    public function getAcceptPartialAuthorization(): ?bool
    {
        return $this->acceptPartialAuthorization;
    }

    /**
     * @param ?bool $value
     */
    public function setAcceptPartialAuthorization(?bool $value = null): self
    {
        $this->acceptPartialAuthorization = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getBuyerEmailAddress(): ?string
    {
        return $this->buyerEmailAddress;
    }

    /**
     * @param ?string $value
     */
    public function setBuyerEmailAddress(?string $value = null): self
    {
        $this->buyerEmailAddress = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getBuyerPhoneNumber(): ?string
    {
        return $this->buyerPhoneNumber;
    }

    /**
     * @param ?string $value
     */
    public function setBuyerPhoneNumber(?string $value = null): self
    {
        $this->buyerPhoneNumber = $value;
        return $this;
    }

    /**
     * @return ?Address
     */
    public function getBillingAddress(): ?Address
    {
        return $this->billingAddress;
    }

    /**
     * @param ?Address $value
     */
    public function setBillingAddress(?Address $value = null): self
    {
        $this->billingAddress = $value;
        return $this;
    }

    /**
     * @return ?Address
     */
    public function getShippingAddress(): ?Address
    {
        return $this->shippingAddress;
    }

    /**
     * @param ?Address $value
     */
    public function setShippingAddress(?Address $value = null): self
    {
        $this->shippingAddress = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getNote(): ?string
    {
        return $this->note;
    }

    /**
     * @param ?string $value
     */
    public function setNote(?string $value = null): self
    {
        $this->note = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getStatementDescriptionIdentifier(): ?string
    {
        return $this->statementDescriptionIdentifier;
    }

    /**
     * @param ?string $value
     */
    public function setStatementDescriptionIdentifier(?string $value = null): self
    {
        $this->statementDescriptionIdentifier = $value;
        return $this;
    }

    /**
     * @return ?CashPaymentDetails
     */
    public function getCashDetails(): ?CashPaymentDetails
    {
        return $this->cashDetails;
    }

    /**
     * @param ?CashPaymentDetails $value
     */
    public function setCashDetails(?CashPaymentDetails $value = null): self
    {
        $this->cashDetails = $value;
        return $this;
    }

    /**
     * @return ?ExternalPaymentDetails
     */
    public function getExternalDetails(): ?ExternalPaymentDetails
    {
        return $this->externalDetails;
    }

    /**
     * @param ?ExternalPaymentDetails $value
     */
    public function setExternalDetails(?ExternalPaymentDetails $value = null): self
    {
        $this->externalDetails = $value;
        return $this;
    }

    /**
     * @return ?CustomerDetails
     */
    public function getCustomerDetails(): ?CustomerDetails
    {
        return $this->customerDetails;
    }

    /**
     * @param ?CustomerDetails $value
     */
    public function setCustomerDetails(?CustomerDetails $value = null): self
    {
        $this->customerDetails = $value;
        return $this;
    }

    /**
     * @return ?OfflinePaymentDetails
     */
    public function getOfflinePaymentDetails(): ?OfflinePaymentDetails
    {
        return $this->offlinePaymentDetails;
    }

    /**
     * @param ?OfflinePaymentDetails $value
     */
    public function setOfflinePaymentDetails(?OfflinePaymentDetails $value = null): self
    {
        $this->offlinePaymentDetails = $value;
        return $this;
    }
}
