<?php

namespace Square\Types;

use Square\Core\Json\JsonSerializableType;
use Square\Core\Json\JsonProperty;
use Square\Core\Types\ArrayType;
use Square\Core\Types\Union;

/**
 * Represents a tax that applies to one or more line item in the order.
 *
 * Fixed-amount, order-scoped taxes are distributed across all non-zero line item totals.
 * The amount distributed to each line item is relative to the amount the item
 * contributes to the order subtotal.
 */
class OrderLineItemTax extends JsonSerializableType
{
    /**
     * @var ?string $uid A unique ID that identifies the tax only within this order.
     */
    #[JsonProperty('uid')]
    private ?string $uid;

    /**
     * @var ?string $catalogObjectId The catalog object ID referencing [CatalogTax](entity:CatalogTax).
     */
    #[JsonProperty('catalog_object_id')]
    private ?string $catalogObjectId;

    /**
     * @var ?int $catalogVersion The version of the catalog object that this tax references.
     */
    #[JsonProperty('catalog_version')]
    private ?int $catalogVersion;

    /**
     * @var ?string $name The tax's name.
     */
    #[JsonProperty('name')]
    private ?string $name;

    /**
     * Indicates the calculation method used to apply the tax.
     * See [OrderLineItemTaxType](#type-orderlineitemtaxtype) for possible values
     *
     * @var ?value-of<OrderLineItemTaxType> $type
     */
    #[JsonProperty('type')]
    private ?string $type;

    /**
     * The percentage of the tax, as a string representation of a decimal
     * number. For example, a value of `"7.25"` corresponds to a percentage of
     * 7.25%.
     *
     * @var ?string $percentage
     */
    #[JsonProperty('percentage')]
    private ?string $percentage;

    /**
     * Application-defined data attached to this tax. Metadata fields are intended
     * to store descriptive references or associations with an entity in another system or store brief
     * information about the object. Square does not process this field; it only stores and returns it
     * in relevant API calls. Do not use metadata to store any sensitive information (such as personally
     * identifiable information or card details).
     *
     * Keys written by applications must be 60 characters or less and must be in the character set
     * `[a-zA-Z0-9_-]`. Entries can also include metadata generated by Square. These keys are prefixed
     * with a namespace, separated from the key with a ':' character.
     *
     * Values have a maximum length of 255 characters.
     *
     * An application can have up to 10 entries per metadata field.
     *
     * Entries written by applications are private and can only be read or modified by the same
     * application.
     *
     * For more information, see [Metadata](https://developer.squareup.com/docs/build-basics/metadata).
     *
     * @var ?array<string, ?string> $metadata
     */
    #[JsonProperty('metadata'), ArrayType(['string' => new Union('string', 'null')])]
    private ?array $metadata;

    /**
     * The amount of money applied to the order by the tax.
     *
     * - For percentage-based taxes, `applied_money` is the money
     * calculated using the percentage.
     *
     * @var ?Money $appliedMoney
     */
    #[JsonProperty('applied_money')]
    private ?Money $appliedMoney;

    /**
     * Indicates the level at which the tax applies. For `ORDER` scoped taxes,
     * Square generates references in `applied_taxes` on all order line items that do
     * not have them. For `LINE_ITEM` scoped taxes, the tax only applies to line items
     * with references in their `applied_taxes` field.
     *
     * This field is immutable. To change the scope, you must delete the tax and
     * re-add it as a new tax.
     * See [OrderLineItemTaxScope](#type-orderlineitemtaxscope) for possible values
     *
     * @var ?value-of<OrderLineItemTaxScope> $scope
     */
    #[JsonProperty('scope')]
    private ?string $scope;

    /**
     * Determines whether the tax was automatically applied to the order based on
     * the catalog configuration. For an example, see
     * [Automatically Apply Taxes to an Order](https://developer.squareup.com/docs/orders-api/apply-taxes-and-discounts/auto-apply-taxes).
     *
     * @var ?bool $autoApplied
     */
    #[JsonProperty('auto_applied')]
    private ?bool $autoApplied;

    /**
     * @param array{
     *   uid?: ?string,
     *   catalogObjectId?: ?string,
     *   catalogVersion?: ?int,
     *   name?: ?string,
     *   type?: ?value-of<OrderLineItemTaxType>,
     *   percentage?: ?string,
     *   metadata?: ?array<string, ?string>,
     *   appliedMoney?: ?Money,
     *   scope?: ?value-of<OrderLineItemTaxScope>,
     *   autoApplied?: ?bool,
     * } $values
     */
    public function __construct(
        array $values = [],
    ) {
        $this->uid = $values['uid'] ?? null;
        $this->catalogObjectId = $values['catalogObjectId'] ?? null;
        $this->catalogVersion = $values['catalogVersion'] ?? null;
        $this->name = $values['name'] ?? null;
        $this->type = $values['type'] ?? null;
        $this->percentage = $values['percentage'] ?? null;
        $this->metadata = $values['metadata'] ?? null;
        $this->appliedMoney = $values['appliedMoney'] ?? null;
        $this->scope = $values['scope'] ?? null;
        $this->autoApplied = $values['autoApplied'] ?? null;
    }

    /**
     * @return ?string
     */
    public function getUid(): ?string
    {
        return $this->uid;
    }

    /**
     * @param ?string $value
     */
    public function setUid(?string $value = null): self
    {
        $this->uid = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getCatalogObjectId(): ?string
    {
        return $this->catalogObjectId;
    }

    /**
     * @param ?string $value
     */
    public function setCatalogObjectId(?string $value = null): self
    {
        $this->catalogObjectId = $value;
        return $this;
    }

    /**
     * @return ?int
     */
    public function getCatalogVersion(): ?int
    {
        return $this->catalogVersion;
    }

    /**
     * @param ?int $value
     */
    public function setCatalogVersion(?int $value = null): self
    {
        $this->catalogVersion = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @param ?string $value
     */
    public function setName(?string $value = null): self
    {
        $this->name = $value;
        return $this;
    }

    /**
     * @return ?value-of<OrderLineItemTaxType>
     */
    public function getType(): ?string
    {
        return $this->type;
    }

    /**
     * @param ?value-of<OrderLineItemTaxType> $value
     */
    public function setType(?string $value = null): self
    {
        $this->type = $value;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getPercentage(): ?string
    {
        return $this->percentage;
    }

    /**
     * @param ?string $value
     */
    public function setPercentage(?string $value = null): self
    {
        $this->percentage = $value;
        return $this;
    }

    /**
     * @return ?array<string, ?string>
     */
    public function getMetadata(): ?array
    {
        return $this->metadata;
    }

    /**
     * @param ?array<string, ?string> $value
     */
    public function setMetadata(?array $value = null): self
    {
        $this->metadata = $value;
        return $this;
    }

    /**
     * @return ?Money
     */
    public function getAppliedMoney(): ?Money
    {
        return $this->appliedMoney;
    }

    /**
     * @param ?Money $value
     */
    public function setAppliedMoney(?Money $value = null): self
    {
        $this->appliedMoney = $value;
        return $this;
    }

    /**
     * @return ?value-of<OrderLineItemTaxScope>
     */
    public function getScope(): ?string
    {
        return $this->scope;
    }

    /**
     * @param ?value-of<OrderLineItemTaxScope> $value
     */
    public function setScope(?string $value = null): self
    {
        $this->scope = $value;
        return $this;
    }

    /**
     * @return ?bool
     */
    public function getAutoApplied(): ?bool
    {
        return $this->autoApplied;
    }

    /**
     * @param ?bool $value
     */
    public function setAutoApplied(?bool $value = null): self
    {
        $this->autoApplied = $value;
        return $this;
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        return $this->toJson();
    }
}
